set more off
clear all
set type double
* Change working directory: "STATA Code for SDG Economy"

********************************************************************************
********************************************************************************
********************************************************************************
**** STATA Code for SDG Economy
**** Building the SDG economy: Needs, spending, and financing for universal achievement of the Sustainable Development Goals
**** Homi Kharas & John McArthur
**** Final data and do-files compiled by Selen Özdoğan
**** January 2020
********************************************************************************
**** 3. Estimate needs gaps between SDG public spending needs and projected expenditures in 2025
********************************************************************************
********************************************************************************
********************************************************************************


* Note: All negative gap values are replaced with 0

/*
	NAMING CONVENTIONS: 
	a_   = filled in missing values with interpolation
	_alt = applied 1.13x GDP pc multiplier to later years
	l_   = natural log 
*/

********************************************************************************
**** Outline *******************************************************************
********************************************************************************
* 1. Gaps in 2025
*		A . Gaps without the 1.13 multiplier
*		B . Gaps with the 1.13 multiplier
*		C . Gaps as % of GDP
* 2. Gaps with extra X% of GDP in DRM
* 		A . Contribution of extra 1% DRM effort


* Merging spending and needs datasets
use "output\merged_full_totals_cons.dta", clear

preserve
use "output/needs_combine_full_cons.dta", clear

keep countryname countrycode pc_total_need lpc_total_need pop2025 gdp_traj2025 gdp_traj2015 pc_ag_need1530 pc_educ_needs pc_health_need2030 pc_pgtSS_need2025 pc_energy_needs2025 pc_flood_needs2025 pc_transport_needs2025 pc_wash_need2025 pc_biodiversity_needs pc_justice_needs

tempfile needs
save `needs'.dta, replace
restore

merge 1:1 countrycode using `needs'.dta
drop _merge
recast str3 countrycode
recast str34 countryname

drop if incomegroup2015=="H"

********************************************************************************
********************************************************************************
*** 1. Gaps in 2025
********************************************************************************
********************************************************************************

********************************************************************************
** A . Gaps without the 1.13 multiplier
********************************************************************************

gen gap_pc_2025 = pc_total_need - a_pc_cntytolgdp_2025
 label var gap_pc_2025 "Per capita gap, 2025 (w/out multiplier)"
count if gap_pc_2025<=0
replace gap_pc_2025 = 0 if gap_pc_2025 <0

gen gap_total_2025=gap_pc_2025*pop2025
 label var gap_total_2025 "Total gap, 2025 (w/out multiplier)"


********************************************************************************
** B . Gaps with the 1.13 multiplier
********************************************************************************

gen gap_pc_2025_alt = pc_total_need - a_pc_cntytolgdp_2025_alt
 label var gap_pc_2025_alt "Per capita gap, 2025 (w/ multiplier)"
count if gap_pc_2025_alt<=0
replace gap_pc_2025_alt = 0 if gap_pc_2025_alt <0

gen gap_total_2025_alt = gap_pc_2025_alt*pop2025
 label var gap_total_2025_alt "Total gap, 2025 (w/ multiplier)"

********************************************************************************
** C . Gaps as % of GDP
********************************************************************************

gen gap_shGDP_2025= gap_pc_2025/pc_gdp_traj2025
 label var gap_shGDP_2025 "Gap in 2025, w/out multiplier (% of GDP)"
list countryname gap_shGDP_2025 if gap_shGDP_2025<0.01

gen gap_shGDP_2025_alt=gap_pc_2025_alt/pc_gdp_traj2025
 label var gap_shGDP_2025_alt "Gap in 2025, w/ multiplier (% of GDP)"
list countryname gap_shGDP_2025_alt if gap_shGDP_2025_alt<0.01

********************************************************************************
********************************************************************************
*** 2. Gaps with extra X% of GDP in DRM
********************************************************************************
********************************************************************************

gen spending_shGDP_2025 = a_cntytolgdp_2025/gdp_traj2025
 label var spending_shGDP_2025 "Spending (% of GDP) in 2025, w/out multiplier"
gen spending_shGDP_2025_alt=a_cntytolgdp_2025_alt/gdp_traj2025
 label var spending_shGDP_2025_alt "Spending (% of GDP) in 2025, w/ multiplier"

forval x = 0.01(0.01)0.05{
local y = `x'*100

gen spending_shGDP_2025_`y' = spending_shGDP_2025 + `x'
 label var spending_shGDP_2025_`y' "Spending (% of GDP) plus `y' %, w/out multiplier, 2025"

gen spending_shGDP_2025_alt_`y' = spending_shGDP_2025_alt + `x'
 label var spending_shGDP_2025_alt_`y' "Spending (% of GDP) plus `y' %, w/ multiplier, 2025"

gen a_pc_cntytolgdp_2025_`y' = (spending_shGDP_2025_`y'*gdp_traj2025)/pop2025
 label var a_pc_cntytolgdp_2025_`y' "Per capita spending plus `y' %, w/out multiplier, 2025"

gen a_cntytolgdp_2025_`y' = spending_shGDP_2025_`y'*gdp_traj2025
 label var a_cntytolgdp_2025_`y' "Total spending plus `y' %, w/out multiplier, 2025"

gen a_pc_cntytolgdp_2025_alt_`y' = (spending_shGDP_2025_alt_`y'*gdp_traj2025)/pop2025
 label var a_pc_cntytolgdp_2025_alt_`y' "Per capita spending plus `y' %, w/ multiplier, 2025"

gen a_cntytolgdp_2025_alt_`y' = spending_shGDP_2025_alt_`y'*gdp_traj2025
 label var a_cntytolgdp_2025_alt_`y' "Total spending plus `y' %, w/ multiplier, 2025"

gen gap_pc_2025_`y' = pc_total_need - a_pc_cntytolgdp_2025_`y'
 label var gap_pc_2025_`y' "Per capita gap plus `y' %, w/out multiplier, 2025"
replace gap_pc_2025_`y' = 0 if gap_pc_2025_`y' <0

gen gap_pc_2025_alt_`y' = pc_total_need - a_pc_cntytolgdp_2025_alt_`y'
 label var gap_pc_2025_alt_`y' "Per capita gap plus `y' %, w/ multiplier, 2025"
replace gap_pc_2025_alt_`y' = 0 if gap_pc_2025_alt_`y' <0

gen gap_total_2025_`y' = gap_pc_2025_`y'*pop2025
 label var gap_total_2025_`y' "Total gap, plus `y' %, w/out multiplier, 2025"

gen gap_total_2025_alt_`y' = gap_pc_2025_alt_`y'*pop2025
 label var gap_total_2025_alt_`y' "Total gap, plus `y' %, w/ multiplier, 2025"

* Tables of averages and total gaps by income group
di "Plus `y' percent"
table incomegroup2015, c(sum gap_total_2025_`y') format(%10.0g)
table incomegroup2015 if gap_pc_2025_`y'!=0 [aweight = pop2025], c(mean gap_pc_2025_`y')
table incomegroup2015, c(sum gap_total_2025_alt_`y') format(%10.0g)
table incomegroup2015 if gap_pc_2025_alt_`y'!=0 [aweight = pop2025], c(mean gap_pc_2025_alt_`y')
}

********************************************************************************
** A . Contribution of extra 1% DRM effort
********************************************************************************

gen gap_contribution_2025=gap_total_2025-gap_total_2025_1
 label var gap_contribution_2025 "Contribution of 1% extra DRM to gap, w/out multiplier, 2025"
replace gap_contribution_2025=gap_total_2025 if gap_shGDP_2025<0.01

gen gap_contribution_2025_alt=gap_total_2025_alt-gap_total_2025_alt_1
 label var gap_contribution_2025_alt "Contribution of 1% extra DRM to gap, w/ multiplier, 2025"
replace gap_contribution_2025_alt=gap_total_2025_alt if gap_shGDP_2025_alt<0.01

save "output/gaps.dta", replace